import { useState } from 'react';
import { Prize } from '@/lib/prizes';

interface LuckyWheelProps {
  prizes: Prize[];
  onSpin: (prize: Prize) => void;
  isSpinning: boolean;
}

export default function LuckyWheel({ prizes, onSpin, isSpinning }: LuckyWheelProps) {
  const [rotation, setRotation] = useState(0);

  const handleSpin = () => {
    if (isSpinning) return;

    // Calculate random rotation (multiple full rotations + random angle)
    const spins = 5 + Math.random() * 5; // 5-10 full rotations
    const finalAngle = Math.random() * 360;
    const totalRotation = rotation + (spins * 360) + finalAngle;
    
    setRotation(totalRotation);

    // Calculate which prize was selected based on final angle
    const normalizedAngle = (360 - (finalAngle % 360)) % 360;
    const prizeIndex = Math.floor((normalizedAngle / 360) * prizes.length);
    const selectedPrize = prizes[prizeIndex];

    // Call onSpin after animation completes
    setTimeout(() => {
      onSpin(selectedPrize);
    }, 3000);
  };

  const segmentAngle = 360 / prizes.length;

  return (
    <div className="relative flex items-center justify-center">
      {/* Wheel Container */}
      <div className="relative">
        {/* Wheel */}
        <div 
          className="w-80 h-80 rounded-full border-8 border-yellow-400 relative overflow-hidden shadow-2xl"
          style={{
            transform: `rotate(${rotation}deg)`,
            transition: isSpinning ? 'transform 3s cubic-bezier(0.23, 1, 0.32, 1)' : 'none'
          }}
        >
          {prizes.map((prize, index) => {
            const startAngle = index * segmentAngle;
            const endAngle = (index + 1) * segmentAngle;
            const midAngle = (startAngle + endAngle) / 2;
            
            return (
              <div
                key={prize.id}
                className="absolute w-full h-full"
                style={{
                  transform: `rotate(${startAngle}deg)`,
                  clipPath: `polygon(50% 50%, 50% 0%, ${50 + 50 * Math.cos((segmentAngle * Math.PI) / 180)}% ${50 - 50 * Math.sin((segmentAngle * Math.PI) / 180)}%)`
                }}
              >
                <div 
                  className="w-full h-full flex items-center justify-center"
                  style={{ backgroundColor: prize.color }}
                >
                  <span 
                    className="text-white font-bold text-sm transform -rotate-90 origin-center"
                    style={{
                      transform: `rotate(${midAngle - 90}deg) translateY(-60px)`
                    }}
                  >
                    {prize.name}
                  </span>
                </div>
              </div>
            );
          })}
          
          {/* Center Circle */}
          <div className="absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2 w-16 h-16 bg-gradient-to-br from-yellow-400 to-yellow-600 rounded-full border-4 border-yellow-300 flex items-center justify-center shadow-lg">
            <span className="text-red-800 font-bold text-xs">SPIN</span>
          </div>
        </div>

        {/* Pointer */}
        <div className="absolute top-0 left-1/2 transform -translate-x-1/2 -translate-y-2 z-10">
          <div className="w-0 h-0 border-l-4 border-r-4 border-b-8 border-l-transparent border-r-transparent border-b-red-600"></div>
        </div>
      </div>

      {/* Spin Button */}
      <button
        onClick={handleSpin}
        disabled={isSpinning}
        className="absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2 w-16 h-16 bg-gradient-to-br from-yellow-400 to-yellow-600 rounded-full border-4 border-yellow-300 flex items-center justify-center shadow-lg hover:from-yellow-300 hover:to-yellow-500 disabled:opacity-50 disabled:cursor-not-allowed z-20"
      >
        <span className="text-red-800 font-bold text-xs">
          {isSpinning ? '...' : 'SPIN'}
        </span>
      </button>
    </div>
  );
}