import { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card, CardContent } from '@/components/ui/card';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import LuckyWheel from '@/components/LuckyWheel';
import { Prize, getPrizes } from '@/lib/prizes';
import { Sparkles, Crown, Zap } from 'lucide-react';

export default function Index() {
  const [username, setUsername] = useState('');
  const [ticketCode, setTicketCode] = useState('');
  const [isSpinning, setIsSpinning] = useState(false);
  const [showResult, setShowResult] = useState(false);
  const [winningPrize, setWinningPrize] = useState<Prize | null>(null);
  const [prizes] = useState<Prize[]>(getPrizes());

  const handleSpin = () => {
    if (!username.trim() || !ticketCode.trim()) {
      alert('Mohon isi Username dan Kode Tiket terlebih dahulu!');
      return;
    }

    setIsSpinning(true);
  };

  const handleSpinComplete = (prize: Prize) => {
    setWinningPrize(prize);
    setIsSpinning(false);
    setShowResult(true);
  };

  const handleCloseResult = () => {
    setShowResult(false);
    setUsername('');
    setTicketCode('');
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-red-900 via-red-800 to-red-900 relative overflow-hidden">
      {/* Animated Background Elements */}
      <div className="absolute inset-0 overflow-hidden">
        {/* Floating coins */}
        <div className="absolute top-20 left-10 text-yellow-400 text-6xl animate-bounce delay-100">💰</div>
        <div className="absolute top-40 right-20 text-yellow-400 text-5xl animate-bounce delay-300">🎰</div>
        <div className="absolute bottom-32 left-20 text-yellow-400 text-7xl animate-pulse delay-500">💎</div>
        <div className="absolute bottom-20 right-32 text-yellow-400 text-6xl animate-bounce delay-700">🏆</div>
        
        {/* Glowing orbs */}
        <div className="absolute top-32 left-1/4 w-4 h-4 bg-yellow-400 rounded-full animate-ping opacity-75"></div>
        <div className="absolute top-64 right-1/3 w-3 h-3 bg-red-400 rounded-full animate-ping opacity-50 delay-1000"></div>
        <div className="absolute bottom-48 left-1/3 w-5 h-5 bg-yellow-300 rounded-full animate-pulse opacity-60"></div>
        
        {/* Sparkle effects */}
        <div className="absolute top-16 right-16 text-yellow-300 text-2xl animate-spin">✨</div>
        <div className="absolute bottom-40 left-16 text-yellow-300 text-3xl animate-spin delay-500">⭐</div>
        <div className="absolute top-1/2 left-8 text-yellow-300 text-2xl animate-pulse delay-1000">💫</div>
      </div>

      {/* Header with enhanced styling */}
      <div className="relative z-10 p-6">
        <div className="text-center mb-8">
          <div className="inline-flex items-center gap-3 bg-gradient-to-r from-yellow-400 to-yellow-600 text-red-900 px-8 py-4 rounded-full shadow-2xl border-4 border-yellow-300">
            <Crown className="w-8 h-8" />
            <h1 className="text-4xl font-bold tracking-wider">SLOT GACOR</h1>
            <Crown className="w-8 h-8" />
          </div>
          <div className="mt-4 flex justify-center gap-4">
            <div className="flex items-center gap-2 bg-red-700/50 px-4 py-2 rounded-full border border-yellow-400">
              <Sparkles className="w-4 h-4 text-yellow-400" />
              <span className="text-yellow-400 font-bold">JACKPOT READY</span>
            </div>
            <div className="flex items-center gap-2 bg-red-700/50 px-4 py-2 rounded-full border border-yellow-400">
              <Zap className="w-4 h-4 text-yellow-400" />
              <span className="text-yellow-400 font-bold">MEGA WIN</span>
            </div>
          </div>
        </div>

        <div className="flex flex-col lg:flex-row items-center justify-center gap-12 max-w-7xl mx-auto">
          {/* Lucky Wheel with enhanced container */}
          <div className="flex-1 flex justify-center">
            <div className="relative">
              {/* Glow effect around wheel */}
              <div className="absolute inset-0 bg-yellow-400/20 rounded-full blur-xl scale-110 animate-pulse"></div>
              <div className="relative">
                <LuckyWheel 
                  prizes={prizes}
                  onSpin={handleSpinComplete}
                  isSpinning={isSpinning}
                />
              </div>
              {/* Decorative elements around wheel */}
              <div className="absolute -top-8 -left-8 text-yellow-400 text-4xl animate-spin">⚡</div>
              <div className="absolute -top-8 -right-8 text-yellow-400 text-4xl animate-spin delay-500">⚡</div>
              <div className="absolute -bottom-8 -left-8 text-yellow-400 text-4xl animate-spin delay-1000">⚡</div>
              <div className="absolute -bottom-8 -right-8 text-yellow-400 text-4xl animate-spin delay-1500">⚡</div>
            </div>
          </div>

          {/* Enhanced Form Section */}
          <div className="flex-1 max-w-md">
            <div className="text-center mb-8">
              <div className="relative">
                <h2 className="text-6xl font-bold text-transparent bg-clip-text bg-gradient-to-r from-yellow-400 via-yellow-300 to-yellow-500 mb-4 tracking-wider animate-pulse">
                  SLOT GACOR
                </h2>
                <div className="absolute -top-2 -left-2 text-red-300 text-8xl font-bold opacity-20 -z-10">
                  777
                </div>
              </div>
              <div className="flex justify-center gap-2 mb-4">
                <div className="w-3 h-3 bg-yellow-400 rounded-full animate-bounce"></div>
                <div className="w-3 h-3 bg-yellow-400 rounded-full animate-bounce delay-100"></div>
                <div className="w-3 h-3 bg-yellow-400 rounded-full animate-bounce delay-200"></div>
              </div>
            </div>

            <Card className="bg-gradient-to-br from-red-800/80 to-red-900/80 border-2 border-yellow-400 backdrop-blur-sm shadow-2xl">
              <CardContent className="p-8">
                <div className="text-center mb-8">
                  <div className="inline-flex items-center gap-2 bg-gradient-to-r from-yellow-500 to-yellow-600 text-red-900 py-3 px-6 rounded-lg border-2 border-yellow-300 shadow-lg">
                    <Sparkles className="w-5 h-5" />
                    <h3 className="text-xl font-bold">MASUKKAN KODE TIKET</h3>
                    <Sparkles className="w-5 h-5" />
                  </div>
                </div>

                <div className="space-y-6">
                  <div className="relative">
                    <Label htmlFor="username" className="text-yellow-300 font-bold text-lg mb-2 block">
                      👤 Username
                    </Label>
                    <Input
                      id="username"
                      type="text"
                      value={username}
                      onChange={(e) => setUsername(e.target.value)}
                      className="bg-white/95 border-2 border-yellow-400 text-gray-800 placeholder-gray-500 h-14 text-center font-bold text-lg rounded-lg shadow-inner focus:border-yellow-300 focus:ring-2 focus:ring-yellow-300"
                      placeholder="Masukkan Username Anda"
                      disabled={isSpinning}
                    />
                  </div>

                  <div className="relative">
                    <Label htmlFor="ticketCode" className="text-yellow-300 font-bold text-lg mb-2 block">
                      🎫 Kode Tiket
                    </Label>
                    <Input
                      id="ticketCode"
                      type="text"
                      value={ticketCode}
                      onChange={(e) => setTicketCode(e.target.value)}
                      className="bg-white/95 border-2 border-yellow-400 text-gray-800 placeholder-gray-500 h-14 text-center font-bold text-lg rounded-lg shadow-inner focus:border-yellow-300 focus:ring-2 focus:ring-yellow-300"
                      placeholder="Masukkan Kode Tiket"
                      disabled={isSpinning}
                    />
                  </div>

                  <Button
                    onClick={handleSpin}
                    disabled={isSpinning}
                    className="w-full bg-gradient-to-r from-yellow-500 via-yellow-600 to-yellow-500 hover:from-yellow-400 hover:via-yellow-500 hover:to-yellow-400 text-red-900 font-bold text-2xl h-16 rounded-xl shadow-2xl disabled:opacity-50 border-2 border-yellow-300 transform hover:scale-105 transition-all duration-200"
                  >
                    {isSpinning ? (
                      <div className="flex items-center gap-2">
                        <div className="w-6 h-6 border-3 border-red-900 border-t-transparent rounded-full animate-spin"></div>
                        SPINNING...
                      </div>
                    ) : (
                      <div className="flex items-center gap-2">
                        <Zap className="w-6 h-6" />
                        SPIN NOW!
                        <Zap className="w-6 h-6" />
                      </div>
                    )}
                  </Button>
                </div>

                {/* Lucky message */}
                <div className="text-center mt-6">
                  <p className="text-yellow-300 text-sm animate-pulse">
                    🍀 Keberuntungan menanti Anda! 🍀
                  </p>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>

      {/* Enhanced Result Dialog */}
      <Dialog open={showResult} onOpenChange={setShowResult}>
        <DialogContent className="bg-gradient-to-br from-yellow-400 via-yellow-500 to-yellow-600 border-4 border-red-600 text-red-900 max-w-md">
          <DialogHeader>
            <DialogTitle className="text-3xl font-bold text-center flex items-center justify-center gap-2">
              <Crown className="w-8 h-8" />
              SELAMAT!
              <Crown className="w-8 h-8" />
            </DialogTitle>
          </DialogHeader>
          <div className="text-center py-8">
            <div className="text-8xl mb-6 animate-bounce">
              {winningPrize?.name === 'ZONK' ? '😔' : '🎉'}
            </div>
            <p className="text-xl font-bold mb-4">
              {winningPrize?.name === 'ZONK' ? 'Jangan menyerah!' : 'Anda mendapatkan:'}
            </p>
            <div 
              className="text-5xl font-bold py-6 px-8 rounded-2xl border-4 border-red-600 inline-block shadow-2xl"
              style={{ backgroundColor: winningPrize?.color }}
            >
              <span className="text-white drop-shadow-lg">
                {winningPrize?.name}
              </span>
            </div>
            <div className="mt-6 space-y-2">
              <p className="text-red-800 font-bold">👤 Username: {username}</p>
              <p className="text-red-800 font-bold">🎫 Kode Tiket: {ticketCode}</p>
            </div>
            {winningPrize?.name !== 'ZONK' && (
              <div className="mt-4 text-sm text-red-700">
                🎊 Hubungi admin untuk claim hadiah! 🎊
              </div>
            )}
          </div>
          <div className="flex justify-center">
            <Button 
              onClick={handleCloseResult}
              className="bg-red-600 hover:bg-red-700 text-white font-bold px-8 py-3 rounded-xl shadow-lg"
            >
              {winningPrize?.name === 'ZONK' ? 'Coba Lagi' : 'Tutup'}
            </Button>
          </div>
        </DialogContent>
      </Dialog>
    </div>
  );
}