import { useState } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Card, CardContent } from '@/components/ui/card';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import LuckyWheel from '@/components/LuckyWheel';
import { Prize, getPrizes, getRandomPrize } from '@/lib/prizes';
import { Settings } from 'lucide-react';

export default function Index() {
  const [username, setUsername] = useState('');
  const [ticketCode, setTicketCode] = useState('');
  const [isSpinning, setIsSpinning] = useState(false);
  const [showResult, setShowResult] = useState(false);
  const [winningPrize, setWinningPrize] = useState<Prize | null>(null);
  const [prizes] = useState<Prize[]>(getPrizes());

  const handleSpin = () => {
    if (!username.trim() || !ticketCode.trim()) {
      alert('Mohon isi Username dan Kode Tiket terlebih dahulu!');
      return;
    }

    setIsSpinning(true);
  };

  const handleSpinComplete = (prize: Prize) => {
    setWinningPrize(prize);
    setIsSpinning(false);
    setShowResult(true);
  };

  const handleCloseResult = () => {
    setShowResult(false);
    setUsername('');
    setTicketCode('');
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-red-900 via-red-800 to-red-900 relative overflow-hidden">
      {/* Background Pattern */}
      <div className="absolute inset-0 opacity-10">
        <div className="absolute top-10 left-10 text-red-300 text-8xl font-bold transform rotate-12">777</div>
        <div className="absolute top-32 right-20 text-red-300 text-6xl font-bold transform -rotate-12">♠</div>
        <div className="absolute bottom-20 left-20 text-red-300 text-7xl font-bold transform rotate-45">♦</div>
        <div className="absolute bottom-32 right-32 text-red-300 text-5xl font-bold transform -rotate-45">♥</div>
      </div>

      {/* Header */}
      <div className="relative z-10 p-6">
        <div className="flex justify-between items-center mb-8">
          <div className="text-left">
            <h1 className="text-3xl font-bold text-yellow-400 tracking-wider">SLOT GACOR</h1>
          </div>
          <div className="flex gap-2">
            <Button 
              onClick={() => window.location.href = '/admin'}
              className="bg-red-600 hover:bg-red-700 text-white"
            >
              <Settings className="w-4 h-4 mr-2" />
              Admin
            </Button>
            <Button className="bg-red-600 hover:bg-red-700 text-white">
              Masuk
            </Button>
            <Button className="bg-yellow-600 hover:bg-yellow-700 text-red-900">
              Daftar
            </Button>
          </div>
        </div>

        <div className="flex flex-col lg:flex-row items-center justify-center gap-12 max-w-7xl mx-auto">
          {/* Lucky Wheel */}
          <div className="flex-1 flex justify-center">
            <LuckyWheel 
              prizes={prizes}
              onSpin={handleSpinComplete}
              isSpinning={isSpinning}
            />
          </div>

          {/* Form Section */}
          <div className="flex-1 max-w-md">
            <div className="text-center mb-8">
              <h2 className="text-5xl font-bold text-red-100 mb-4 tracking-wider">SLOT GACOR</h2>
            </div>

            <Card className="bg-red-800/50 border-yellow-400 backdrop-blur-sm">
              <CardContent className="p-6">
                <div className="text-center mb-6">
                  <h3 className="text-xl font-bold text-yellow-400 bg-red-700/50 py-2 px-4 rounded-lg border border-yellow-400">
                    MASUKKAN KODE TIKET
                  </h3>
                </div>

                <div className="space-y-4">
                  <div>
                    <Label htmlFor="username" className="text-red-100 font-medium">Username</Label>
                    <Input
                      id="username"
                      type="text"
                      value={username}
                      onChange={(e) => setUsername(e.target.value)}
                      className="bg-white/90 border-red-300 text-gray-800 placeholder-gray-500 h-12 text-center font-medium"
                      placeholder="Username"
                      disabled={isSpinning}
                    />
                  </div>

                  <div>
                    <Label htmlFor="ticketCode" className="text-red-100 font-medium">Kode Tiket</Label>
                    <Input
                      id="ticketCode"
                      type="text"
                      value={ticketCode}
                      onChange={(e) => setTicketCode(e.target.value)}
                      className="bg-white/90 border-red-300 text-gray-800 placeholder-gray-500 h-12 text-center font-medium"
                      placeholder="Kode Tiket"
                      disabled={isSpinning}
                    />
                  </div>

                  <Button
                    onClick={handleSpin}
                    disabled={isSpinning}
                    className="w-full bg-gradient-to-r from-red-600 to-red-700 hover:from-red-700 hover:to-red-800 text-white font-bold text-lg h-14 rounded-lg shadow-lg disabled:opacity-50"
                  >
                    {isSpinning ? 'SPINNING...' : 'SPIN'}
                  </Button>
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>

      {/* Result Dialog */}
      <Dialog open={showResult} onOpenChange={setShowResult}>
        <DialogContent className="bg-gradient-to-br from-red-800 to-red-900 border-yellow-400 text-white">
          <DialogHeader>
            <DialogTitle className="text-2xl font-bold text-yellow-400 text-center">
              Selamat!
            </DialogTitle>
          </DialogHeader>
          <div className="text-center py-6">
            <div className="text-6xl font-bold text-yellow-400 mb-4">
              🎉
            </div>
            <p className="text-xl mb-2">Anda mendapatkan:</p>
            <div 
              className="text-4xl font-bold py-4 px-6 rounded-lg border-2 border-yellow-400 inline-block"
              style={{ backgroundColor: winningPrize?.color }}
            >
              {winningPrize?.name}
            </div>
            <p className="text-red-200 mt-4">Username: {username}</p>
            <p className="text-red-200">Kode Tiket: {ticketCode}</p>
          </div>
          <div className="flex justify-center">
            <Button 
              onClick={handleCloseResult}
              className="bg-yellow-600 hover:bg-yellow-700 text-red-900 font-bold"
            >
              Tutup
            </Button>
          </div>
        </DialogContent>
      </Dialog>
    </div>
  );
}